#!/usr/bin/env python3
"""
Generate a 6×4in OTP booklet PDF from a CSV of symbols.

Usage:
    python3 pdf_bookmaker.py <input_csv> <output_pdf> [--layout portrait|landscape]

    <input_csv>   – CSV file with raw symbols (cards, dice, coins). Blank cells are ignored.
    <output_pdf>  – Destination PDF (overwrites if it exists).
    --layout      – portrait (one page per sheet) or landscape (same; default).

The script creates one 6×4 in OTP page per Letter sheet, draws red crop‑marks,
adds a bold “Page N” header, left‑hand “[Line X]” labels, and a bottom
“TOP SECRET” footer.
"""

import argparse
import csv
import math
import sys
from reportlab.lib import colors
from reportlab.lib.pagesizes import LETTER, landscape, portrait
from reportlab.lib.units import inch
from reportlab.pdfgen import canvas

# ----------------------------------------------------------------------
# Layout constants
# ----------------------------------------------------------------------
PAGE_W_IN = 6.0          # width of OTP page (inches)
PAGE_H_IN = 4.0          # height of OTP page (inches)

SPINE_MARGIN_IN = 0.4   # space for binding on the left edge
COLS_PER_PAGE = 10
ROWS_PER_PAGE = 10

FONT_NAME = "Helvetica-Bold"
FONT_SIZE = 12.5           # Xpt, bold
LINE_SPACING = 1.5       # tighter rows to fit 30 lines

CROP_MARK_LEN_PT = 2
CROP_COLOR = colors.red

# ----------------------------------------------------------------------
def parse_args():
    parser = argparse.ArgumentParser(description="Generate a 6×4 in OTP booklet PDF.")
    parser.add_argument("csv_file", help="CSV file with raw symbols")
    parser.add_argument("pdf_file", help="Output PDF file")
    parser.add_argument(
        "--layout",
        choices=["portrait", "landscape"],
        default="landscape",
        help="One OTP page per Letter sheet (portrait or landscape orientation)",
    )
    return parser.parse_args()

# ----------------------------------------------------------------------
def read_symbols(csv_path):
    """Flatten the CSV into a list of symbols, preserving order."""
    symbols = []
    with open(csv_path, newline="", encoding="utf-8") as f:
        for row in csv.reader(f):
            for cell in row:
                cell = cell.strip()
                if cell:
                    symbols.append(cell)
    return symbols

# ----------------------------------------------------------------------
def chunk_symbols(symbols, per_page):
    """Yield successive slices of `per_page` symbols."""
    for i in range(0, len(symbols), per_page):
        yield symbols[i:i + per_page]

# ----------------------------------------------------------------------
def draw_crop_marks(c, x0, y0, w, h):
    """Draw tiny cross‑hair marks at the four corners of a rectangle."""
    cm = CROP_MARK_LEN_PT
    c.setStrokeColor(CROP_COLOR)
    c.setLineWidth(0.5)
    # lower‑left
    c.line(x0 - cm, y0, x0 + cm, y0)
    c.line(x0, y0 - cm, x0, y0 + cm)
    # lower‑right
    c.line(x0 + w - cm, y0, x0 + w + cm, y0)
    c.line(x0 + w, y0 - cm, x0 + w, y0 + cm)
    # upper‑left
    c.line(x0 - cm, y0 + h, x0 + cm, y0 + h)
    c.line(x0, y0 + h - cm, x0, y0 + h + cm)
    # upper‑right
    c.line(x0 + w - cm, y0 + h, x0 + w + cm, y0 + h)
    c.line(x0 + w, y0 + h - cm, x0 + w, y0 + h + cm)

# ----------------------------------------------------------------------
def render_page(c, symbols, page_number):
    """Draw a single 6×4in OTP page."""
    assert len(symbols) == COLS_PER_PAGE * ROWS_PER_PAGE

    # ----- Header (centered) -----
    header_text = f"Page {page_number}"
    c.setFont(FONT_NAME, FONT_SIZE + 2)          # 18pt header
    header_w = c.stringWidth(header_text, FONT_NAME, FONT_SIZE + 2)
    top_margin = 0.35 * inch                     # space from top edge
    c.drawString((PAGE_W_IN * inch - header_w) / 2,
                 PAGE_H_IN * inch - top_margin,
                 header_text)

    # ----- Grid with line numbers -----
    c.setFont(FONT_NAME, FONT_SIZE)
    line_height = FONT_SIZE * LINE_SPACING

    # start just below the header (add a tiny cushion)
    start_y = PAGE_H_IN * inch - top_margin - 0.3 * inch

    # left margin includes spine + extra gutter
    left_margin = (SPINE_MARGIN_IN + 0.35) * inch
    col_width = (PAGE_W_IN * inch - left_margin) / COLS_PER_PAGE

    for row in range(ROWS_PER_PAGE):
        y = start_y - row * line_height

        # line label – only the bracketed number
        line_label = f"[{row + 1}]"
        c.drawString(0.3 * inch, y, line_label)

        # draw the ten symbols for this row
        for col in range(COLS_PER_PAGE):
            x = left_margin + col * col_width + 0.04 * inch
            idx = row * COLS_PER_PAGE + col
            token = symbols[idx]
            c.drawString(x, y, token)

    # ----- Footer (bottom centre) -----
    footer_text = "TOP SECRET"
    c.setFont(FONT_NAME, FONT_SIZE)
    footer_w = c.stringWidth(footer_text, FONT_NAME, FONT_SIZE)
    c.drawString((PAGE_W_IN * inch - footer_w) / 2,
                 0.45 * inch,
                 footer_text)

    # ----- Crop marks -----
    draw_crop_marks(c, 0, 0, PAGE_W_IN * inch, PAGE_H_IN * inch)

# ----------------------------------------------------------------------
def main():
    args = parse_args()
    symbols = read_symbols(args.csv_file)
    if not symbols:
        sys.exit("ERROR: No symbols found in the CSV file.")

    per_page = COLS_PER_PAGE * ROWS_PER_PAGE
    total_pages = math.ceil(len(symbols) / per_page)
    padded = symbols + [""] * (total_pages * per_page - len(symbols))

    # One OTP page per Letter sheet
    page_size = landscape(LETTER) if args.layout == "landscape" else portrait(LETTER)
    c = canvas.Canvas(args.pdf_file, pagesize=page_size)

    page_num = 1
    for chunk in chunk_symbols(padded, per_page):
        render_page(c, chunk, page_num)
        c.showPage()
        page_num += 1

    c.save()
    print(f"[INFO] Generated {total_pages} OTP pages → {args.pdf_file}")

if __name__ == "__main__":
    main()